<?php
/**
 * Plugin Name: SAR Compare Widget
 * Description: Provides a widget to display SAR comparison data on any website.
 * Version: 1.0
 * Author: Your Name
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Register the REST API endpoint
function register_sar_compare_rest_route() {
    register_rest_route('sar-compare/v1', '/compare/', array(
        'methods' => 'GET',
        'callback' => 'sar_compare_rest_callback',
        'permission_callback' => '__return_true'
    ));
}
add_action('rest_api_init', 'register_sar_compare_rest_route');

function sar_compare_rest_callback($request) {
    global $wpdb;

    $title = $request->get_param('title');
    $pos = $request->get_param('pid');

    if (!$pos && !$title) {
        return new WP_Error('no_params', 'No product IDs or title provided', array('status' => 400));
    }

    if ($title) {
        // Get the current post title
        $current_post_title = strtolower($title);

        // Retrieve all phone models and their corresponding post_id from the sar_data table
        $table_name = $wpdb->prefix . 'sar_data';
        $rows = $wpdb->get_results("SELECT post_id, model FROM $table_name", 'ARRAY_A');

        if (empty($rows)) {
            return new WP_Error('no_models', 'No phone models found in the database.', array('status' => 404));
        }

        // Create an array of models and their post_ids
        $phone_models = [];
        foreach ($rows as $row) {
            $phone_models[strtolower($row['model'])] = ['post_id' => $row['post_id'], 'original_model' => $row['model']];
        }

        // Track the longest match for each starting letter
        $longest_matches = [];

        // Check if the current post title contains any of the phone models
        foreach ($phone_models as $model => $data) {
            if (strpos($current_post_title, $model) !== false) {
                $first_letter = $model[0];
                if (!isset($longest_matches[$first_letter]) || strlen($model) > strlen($longest_matches[$first_letter]['model'])) {
                    $longest_matches[$first_letter] = [
                        'model' => $model,
                        'post_id' => $data['post_id'],
                        'original_model' => $data['original_model']
                    ];
                }
            }
        }

        // Prepare the matched models and post_ids
        $matched_post_ids = [];
        foreach ($longest_matches as $match) {
            $matched_post_ids[] = $match['post_id'];
        }

        $pos = implode('-', $matched_post_ids);
    }

    // Get the list of products to compare
    $compList = explode('-', $pos);
    $compList = array_filter($compList);

    // Check if each post ID is valid
    $table_name = $wpdb->prefix . 'sar_data';
    foreach ($compList as $key => $post_id) {
        if (strlen($post_id) < 7 && is_numeric($post_id)) {
            $post = get_post($post_id);
            $row = $wpdb->get_row("SELECT * FROM $table_name WHERE post_id = '$post_id'", 'ARRAY_A');
            if (!$post || !$row) {
                // Remove invalid post ID from the array
                unset($compList[$key]);
            }
        } else {
            unset($compList[$key]);
        }
    }

    if (empty($compList)) {
        return new WP_Error('invalid_pids', 'Invalid product IDs', array('status' => 400));
    }

    // Fetch product data
    $products = [];
    foreach ($compList as $pid) {
        $title = get_the_title($pid);
        $link = get_permalink($pid);
        $image = get_the_post_thumbnail_url($pid, 'thumbnail');
        $row = $wpdb->get_row("SELECT * FROM $table_name WHERE post_id = '$pid'", 'ARRAY_A');

        if ($row) {
            $products[] = array(
                'title' => $title,
                'link' => $link,
                'image' => $image,
                'sar_data' => array(
                    'ushead' => $row['ushead'],
                    'usbody' => $row['usbody'],
                    'productspecific' => $row['productspecific'],
                    'simulheadus' => $row['simulheadus'],
                    'simulbodyus' => $row['simulbodyus'],
                    'hotspot' => $row['hotspot'],
                )
            );
        }
    }

    return rest_ensure_response($products);
}

// Function to create the JavaScript file for the widget
function sar_compare_create_js_file() {
    $api_url = home_url('/wp-json/sar-compare/v1/compare');
    $js_code = <<<EOD
(function() {
    // Function to create the comparison widget
    function createSarCompareWidget() {
        var apiUrl = '$api_url';
        var container = document.createElement('div');
        container.innerHTML = '<p>Loading SAR comparison data...</p>';
        document.body.appendChild(container);

        var title = document.title;

        fetch(apiUrl + '?title=' + encodeURIComponent(title))
            .then(response => response.json())
            .then(data => {
                if (data.code && data.message) {
                    container.innerHTML = '<p>Error: ' + data.message + '</p>';
                    return;
                }

                var content = '<h3>SAR Comparison</h3><div class="sar-compare-column">';
                data.forEach(product => {
                    content += '<div class="sar-product">';
                    content += '<h4><a href="' + product.link + '" target="_blank">' + product.title + '</a></h4>';
                    content += '<img src="' + product.image + '" alt="' + product.title + '">';
                    content += '<p>Head SAR: ' + product.sar_data.ushead + ' W/kg</p>';
                    content += '<p>Body SAR: ' + product.sar_data.usbody + ' W/kg</p>';
                    content += '</div>';
                });
                content += '</div>';

                container.innerHTML = content;
            })
            .catch(error => {
                container.innerHTML = '<p>Error fetching data: ' + error.message + '</p>';
            });
    }

    // Execute the function to create the widget
    createSarCompareWidget();
})();
EOD;

    file_put_contents(plugin_dir_path(__FILE__) . 'sar-compare-widget.js', $js_code);
}
register_activation_hook(__FILE__, 'sar_compare_create_js_file');

// Function to create the instruction page
function sar_compare_create_instruction_page() {
    $plugin_url = plugin_dir_url(__FILE__) . 'sar-compare-widget.js';
    $page_content = <<<EOD
<h1>How to Add SAR Compare Widget to Your Website</h1>
<p>You can easily display SAR comparison data on your website by including the following script on your page:</p>
<pre><code>&lt;script src="$plugin_url"&gt;&lt;/script&gt;</code></pre>
<p>This script will automatically detect the page title and fetch the SAR comparison data for matching phone models. If the title of the page contains a phone model that we have specs for, the SAR data and specs will be displayed.</p>
<button id="copy-code-button">Copy Code</button>
<script>
document.getElementById('copy-code-button').addEventListener('click', function() {
    var code = '&lt;script src="$plugin_url"&gt;&lt;/script&gt;';
    navigator.clipboard.writeText(code).then(function() {
        alert('Code copied to clipboard!');
    }, function() {
        alert('Failed to copy code.');
    });
});
</script>
EOD;

    $page_id = wp_insert_post(array(
        'post_title' => 'SAR Compare Widget Instructions',
        'post_content' => $page_content,
        'post_status' => 'publish',
        'post_type' => 'page'
    ));

    if ($page_id) {
        update_option('sar_compare_instruction_page_id', $page_id);
    }
}
register_activation_hook(__FILE__, 'sar_compare_create_instruction_page');

// Function to delete the instruction page upon plugin deactivation
function sar_compare_delete_instruction_page() {
    $page_id = get_option('sar_compare_instruction_page_id');
    if ($page_id) {
        wp_delete_post($page_id, true);
        delete_option('sar_compare_instruction_page_id');
    }
}
register_deactivation_hook(__FILE__, 'sar_compare_delete_instruction_page');

// Function to get the URL of the instruction page
function get_sar_compare_instruction_page_url() {
    $page_id = get_option('sar_compare_instruction_page_id');
    if ($page_id) {
        return get_permalink($page_id);
    }
    return '';
}

// Add link to plugin action links
function sar_compare_plugin_action_links($links) {
    $instruction_page_url = get_sar_compare_instruction_page_url();
    if ($instruction_page_url) {
        $links[] = '<a href="' . esc_url($instruction_page_url) . '">Instructions</a>';
    }
    return $links;
}
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'sar_compare_plugin_action_links');
